﻿using System;
using System.Net;
using System.Threading;
using gov.va.med.vbecs.Common.AppServices;
using gov.va.med.vbecs.Common.Log;
using gov.va.med.VBECS.Communication.Common;
using gov.va.med.VBECS.Communication.Server;
using IServer = gov.va.med.vbecs.Common.IServer;

namespace VBECS.Services.Common
{
    /// <summary>
    /// Common class for server that supports TCP IP interfacing
    /// </summary>
    public abstract class InterfaceServer : IServer
    {
        // Internal TCP server
        protected gov.va.med.VBECS.Communication.Server.IServer Server;

        public void Start()
        {
            if (Server != null) return; // The server already started
            var endpoint = get_endpoint();
            Server = ServerFactory.CreateServer(endpoint);

            Server.FatalErrorOccured += delegate(object sender, ThreadExceptionEventArgs args)
            {
                // Report to caller is subscribed
                if (FatalErrorOccured != null)
                    FatalErrorOccured(sender, args);
            };
            Server.ClientDisconnected += server_client_disconnected;
            Server.ClientConnected += server_client_connected;
            Server.Start();
            get_logger().Debug(string.Format("A server started (on ip: {0}, port {1})", endpoint.Address, endpoint.Port));
        }

        public void Stop()
        {
            if (Server == null) return; // The server is not started
            Server.Stop();
            Server = null;
        }

        private void server_client_connected(object sender, ClientDummyEventArgs e)
        {
            get_logger().Debug("A new client is connected. Client Id = " + e.Client.Id);

            //Register to MessageReceived event to receive messages from new client
            e.Client.MessageReceived += client_message_received;
        }

        private void server_client_disconnected(object sender, ClientDummyEventArgs e)
        {
            get_logger().Debug("A client is disconnected! Client Id = " + e.Client.Id);
        }

        /// <summary>
        /// Returns logger object from children classes
        /// </summary>
        /// <returns>logger</returns>
        protected abstract ILogger get_logger();

        /// <summary>
        /// Is called for every received message
        /// </summary>
        protected abstract void client_message_received(object sender, MessageEventArgs e);

        /// <summary>
        /// Is called for every received message
        /// </summary>
        protected virtual IPEndPoint get_endpoint()
        {
            // Read port number from configuration
            var portNumber = (int)GlobalContext.Instance().AppSettingsReader.GetValue("ListenerPortNumber", typeof(int));
            IPAddress listenerIpAddress = null;
            try
            {
#if DEBUG //for debug configuration do not use IP from DB, but use development machine IP instead.
                listenerIpAddress = IPAddress.Any;
#else
                listenerIpAddress = IPAddress.Parse(((string)GlobalContext.Instance().AppSettingsReader.GetValue("ListenerIpAddress", typeof(string))).Trim());
#endif
                return new IPEndPoint(listenerIpAddress, portNumber);
            }
            catch (FormatException)
            {
                get_logger().Error(string.Format("Failed to create a server for this IP: {0}", listenerIpAddress));
                throw;
            }
        }

        /// <summary>
        /// Event for signaling fatal errors to outer funtionality
        /// </summary>
        public event EventHandler<ThreadExceptionEventArgs> FatalErrorOccured;
    }
}



//#if DEBUG //for debug configuration do not use IP from DB, but use development machine IP instead.
//            return new IPEndPoint(IPAddress.Any, portNumber);
//#else
//            var listenerIpAddress = (string)GlobalContext.Instance().AppSettingsReader.GetValue("ListenerIpAddress", typeof(string));
//            return new IPEndPoint(IPAddress.Parse(listenerIpAddress), portNumber);
//#endif